#!/bin/bash

# =================================================================
#  CONFIGURACIÓN Y UTILERÍAS
# =================================================================

# Colores para la terminal
VERDE='\033[0;32m'
ROJO='\033[0;31m'
AZUL='\033[0;34m'
AMARILLO='\033[1;33m'
NORMAL='\033[0m' # Sin color

# Función para mostrar mensajes de estado
function mensaje_estado {
    echo -e "${AZUL}==>${NORMAL} ${AMARILLO}$1${NORMAL}"
}

# Función para mostrar un error
function mensaje_error {
    echo -e "${ROJO}!!! ERROR: $1 ${NORMAL}"
}

# Función para pausar
function pausar {
    read -rp "Presiona [ENTER] para continuar..."
}

# =================================================================
#  FUNCIONES DE INSTALACIÓN
# =================================================================

function instalar_lamp {
    mensaje_estado "Iniciando instalación de la Pila LAMP (Apache, MariaDB, PHP)..."
    
    # 1. Actualizar el sistema
    sudo apt update
    
    # 2. Instalar Apache
    sudo apt install -y apache2
    sudo systemctl enable apache2
    sudo systemctl start apache2
    
    # 3. Instalar MariaDB (Reemplazo moderno y común de MySQL)
    sudo apt install -y mariadb-server
    sudo systemctl enable mariadb
    sudo systemctl start mariadb
    
    # 4. Instalar PHP y módulos comunes
    sudo apt install -y php libapache2-mod-php php-mysql php-cli php-curl php-json php-mbstring php-xml
    
    # 5. Reiniciar Apache para aplicar los cambios de PHP
    sudo systemctl restart apache2
    
    mensaje_estado "LAMP instalado correctamente. Por favor, ejecuta 'sudo mysql_secure_installation' para asegurar MariaDB."
    pausar
}

function instalar_vscode {
    mensaje_estado "Iniciando instalación de Visual Studio Code (vía repositorio APT)..."
    
    # 1. Instalar dependencias
    sudo apt install -y wget apt-transport-https
    
    # 2. Importar la llave GPG de Microsoft
    wget -qO- https://packages.microsoft.com/keys/microsoft.asc | gpg --dearmor > packages.microsoft.gpg
    sudo install -o root -g root -m 644 packages.microsoft.gpg /etc/apt/trusted.gpg.d/
    
    # 3. Agregar el repositorio de VS Code
    sudo sh -c 'echo "deb [arch=amd64,arm64,armhf] https://packages.microsoft.com/repos/code stable main" > /etc/apt/sources.list.d/vscode.list'
    rm packages.microsoft.gpg
    
    # 4. Actualizar la lista de paquetes e instalar VS Code
    sudo apt update
    sudo apt install -y code # El paquete se llama 'code'
    
    mensaje_estado "Visual Studio Code instalado correctamente."
    pausar
}

function instalar_phpmyadmin {
    mensaje_estado "Iniciando instalación y configuración de phpMyAdmin..."
    
    sudo apt install -y phpmyadmin
    
    if [ ! -f /etc/apache2/conf-available/phpmyadmin.conf ]; then
        mensaje_error "El archivo de configuración de phpMyAdmin no se encontró. No se puede continuar con la configuración de Apache."
        pausar
        return
    fi
    
    # Habilitar la configuración de phpMyAdmin en Apache
    sudo ln -s /etc/phpmyadmin/apache.conf /etc/apache2/conf-available/phpmyadmin.conf
    sudo a2enconf phpmyadmin
    
    # Habilitar el módulo de reescritura de Apache
    sudo a2enmod rewrite

    # Reiniciar Apache
    sudo systemctl restart apache2
    
    mensaje_estado "phpMyAdmin instalado. Accede a http://localhost/phpmyadmin o http://[IP_Servidor]/phpmyadmin"
    pausar
}


# =================================================================
#  FUNCIONES DE VIRTUAL HOST (CRUD) - ACTUALIZADA
# =================================================================

VHOSTS_DIR="/etc/apache2/sites-available"
HOSTS_FILE="/etc/hosts"
DOC_ROOT="/var/www"

function crear_virtualhost {
    mensaje_estado "=== Creación de un Nuevo Virtual Host ==="
    read -rp "Nombre del Dominio (ej: mi-sitio.local): " DOMAIN
    
    # Sanitizar el nombre del dominio
    DOMAIN=$(echo "$DOMAIN" | tr '[:upper:]' '[:lower:]' | sed 's/[^a-z0-9\.\-]//g')
    
    if [ -z "$DOMAIN" ]; then
        mensaje_error "Nombre de dominio no válido."
        pausar
        return
    fi
    
    VHOST_CONF="$VHOSTS_DIR/$DOMAIN.conf"
    VHOST_PATH="$DOC_ROOT/$DOMAIN" # Ruta base de la carpeta
    
    if [ -f "$VHOST_CONF" ]; then
        mensaje_error "El Virtual Host '$DOMAIN' ya existe. Intenta con 'Eliminar' o 'Listar'."
        pausar
        return
    fi

    mensaje_estado "Creando directorio raíz del documento: $VHOST_PATH/public_html"
    # Solo crea la carpeta, sin archivos.
    sudo mkdir -p "$VHOST_PATH/public_html" 
    
    # APLICACIÓN DE PERMISOS Y PROPIETARIO
    mensaje_estado "Aplicando permisos (755) y propietario ($USER:$USER) al directorio."
    sudo chmod -R 755 "$VHOST_PATH" 
    sudo chown -R $USER:$USER "$VHOST_PATH" 
    
    
    mensaje_estado "Creando archivo de configuración de Virtual Host: $VHOST_CONF"
    
    # Plantilla de configuración de Virtual Host
    sudo bash -c "cat << EOF > $VHOST_CONF
<VirtualHost *:80>
    ServerAdmin webmaster@$DOMAIN
    ServerName $DOMAIN
    ServerAlias www.$DOMAIN
    DocumentRoot $VHOST_PATH/public_html

    <Directory $VHOST_PATH/public_html>
        Options Indexes FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog \${APACHE_LOG_DIR}/$DOMAIN-error.log
    CustomLog \${APACHE_LOG_DIR}/$DOMAIN-access.log combined
</VirtualHost>
EOF"
    
    # Habilitar el sitio
    sudo a2ensite "$DOMAIN.conf"
    
    # Agregar al archivo hosts (para que funcione localmente)
    if ! grep -q "$DOMAIN" "$HOSTS_FILE"; then
        mensaje_estado "Agregando '$DOMAIN' a $HOSTS_FILE..."
        echo "127.0.0.1 $DOMAIN" | sudo tee -a "$HOSTS_FILE" > /dev/null
    fi
    
    # Reiniciar Apache
    sudo systemctl restart apache2
    
    echo -e "${VERDE}✅ Virtual Host '$DOMAIN' creado y activado.${NORMAL}"
    echo "Carpeta raíz: $VHOST_PATH/public_html (¡VACÍA Y LISTA PARA TU CÓDIGO!)"
    echo "Acceso: http://$DOMAIN"
    pausar
}

function listar_virtualhosts {
    mensaje_estado "=== Virtual Hosts Activos (en $VHOSTS_DIR) ==="
    VHOSTS=$(ls $VHOSTS_DIR/*.conf 2>/dev/null | xargs -n 1 basename | sed 's/.conf$//')
    
    if [ -z "$VHOSTS" ]; then
        echo "No se encontraron Virtual Hosts creados por el script."
    else
        echo -e "${VERDE}Dominio | Estado${NORMAL}"
        echo "----------------------"
        for VHOST in $VHOSTS; do
            if [ -L "/etc/apache2/sites-enabled/$VHOST.conf" ]; then
                STATUS="${VERDE}ACTIVO${NORMAL}"
            else
                STATUS="${ROJO}INACTIVO${NORMAL}"
            fi
            printf "%s | %s\n" "$VHOST" "$STATUS"
        done
        
        mensaje_estado "=== Entradas en /etc/hosts relacionadas ==="
        grep '127.0.0.1 ' "$HOSTS_FILE" | grep -v 'localhost' | sed 's/^127\.0\.0\.1 //g'
    fi
    pausar
}

function eliminar_virtualhost {
    mensaje_estado "=== Eliminación de Virtual Host ==="
    listar_virtualhosts
    
    read -rp "Introduce el nombre exacto del dominio a eliminar: " DOMAIN_TO_DELETE
    
    if [ -z "$DOMAIN_TO_DELETE" ]; then
        mensaje_error "Dominio no especificado."
        pausar
        return
    fi
    
    VHOST_CONF="$VHOSTS_DIR/$DOMAIN_TO_DELETE.conf"
    VHOST_PATH="$DOC_ROOT/$DOMAIN_TO_DELETE"
    
    if [ ! -f "$VHOST_CONF" ]; then
        mensaje_error "El Virtual Host '$DOMAIN_TO_DELETE' no existe."
        pausar
        return
    fi

    read -rp "¿Estás seguro de que quieres ELIMINAR el VHost '$DOMAIN_TO_DELETE' y su carpeta raíz ($VHOST_PATH)? (S/N): " CONFIRM

    if [[ "$CONFIRM" =~ ^[Ss]$ ]]; then
        # 1. Deshabilitar y eliminar archivo de configuración
        sudo a2dissite "$DOMAIN_TO_DELETE.conf"
        sudo rm -f "$VHOST_CONF"
        
        # 2. Eliminar la carpeta raíz del documento
        sudo rm -rf "$VHOST_PATH"
        
        # 3. Eliminar del archivo hosts
        sudo sed -i "/$DOMAIN_TO_DELETE/d" "$HOSTS_FILE"
        
        # 4. Reiniciar Apache
        sudo systemctl restart apache2
        
        echo -e "${VERDE}✅ Virtual Host '$DOMAIN_TO_DELETE' eliminado completamente.${NORMAL}"
    else
        mensaje_estado "Operación de eliminación cancelada."
    fi
    pausar
}

# =================================================================
#  MENÚ PRINCIPAL Y CRUD
# =================================================================

function menu_principal {
    clear
    echo -e "${VERDE}======================================================${NORMAL}"
    echo -e "${AZUL}  GESTOR AUTOMATIZADO DE ENTORNO DE DESARROLLO (APT)  ${NORMAL}"
    echo -e "${VERDE}======================================================${NORMAL}"
    
    PS3="${AMARILLO}Elige una opción (1-5): ${NORMAL}"
    
    options=(
        "Instalar Pila LAMP (Apache, MariaDB, PHP)"
        "Instalar Visual Studio Code"
        "Instalar y Configurar phpMyAdmin"
        "Administrar Virtual Hosts (CRUD)"
        "Salir"
    )
    
    select opt in "${options[@]}"; do
        case $opt in
            "Instalar Pila LAMP (Apache, MariaDB, PHP)")
                instalar_lamp
                menu_principal
                ;;
            "Instalar Visual Studio Code")
                instalar_vscode
                menu_principal
                ;;
            "Instalar y Configurar phpMyAdmin")
                instalar_phpmyadmin
                menu_principal
                ;;
            "Administrar Virtual Hosts (CRUD)")
                menu_crud_vhost
                ;;
            "Salir")
                echo -e "${AZUL}¡Adiós! ¡Feliz codificación!${NORMAL}"
                break
                ;;
            *)
                mensaje_error "Opción inválida $REPLY"
                ;;
        esac
    done
}

function menu_crud_vhost {
    clear
    echo -e "${VERDE}==================================${NORMAL}"
    echo -e "${AZUL}  ADMINISTRACIÓN DE VIRTUAL HOSTS ${NORMAL}"
    echo -e "${VERDE}==================================${NORMAL}"
    
    PS3="${AMARILLO}Elige una acción para Virtual Hosts (1-4): ${NORMAL}"
    
    options=(
        "Crear Nuevo Virtual Host"
        "Listar Virtual Hosts"
        "Eliminar Virtual Host"
        "Volver al Menú Principal"
    )
    
    select opt in "${options[@]}"; do
        case $opt in
            "Crear Nuevo Virtual Host")
                crear_virtualhost
                menu_crud_vhost
                ;;
            "Listar Virtual Hosts")
                listar_virtualhosts
                menu_crud_vhost
                ;;
            "Eliminar Virtual Host")
                eliminar_virtualhost
                menu_crud_vhost
                ;;
            "Volver al Menú Principal")
                menu_principal
                ;;
            *)
                mensaje_error "Opción inválida $REPLY"
                ;;
        esac
    done
}

# =================================================================
#  INICIO DEL SCRIPT
# =================================================================

menu_principal